<?php
date_default_timezone_set('Europe/Berlin'); // Germany time zone

include_once __DIR__ . "/../../connect.php";
include_once __DIR__ . "/../../orders_functions.php";

header('Content-Type: application/json');
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Debug log file
$debugFile = __DIR__ . "/approve_debug.log";
file_put_contents($debugFile, "----- Approve Script Started -----\n", FILE_APPEND);

// Safe POST data
$orderId   = $_POST['ordersid'] ?? null;
$userId    = $_POST['usersid'] ?? null;
$adminId   = $_POST['adminid'] ?? null;
$adminName = $_POST['adminname'] ?? null;
$fcmToken  = $_POST['fcm_token'] ?? null;

// Log POST
file_put_contents($debugFile, "POST received: " . json_encode($_POST) . "\n", FILE_APPEND);

// Validate required fields
if (!$orderId || !$userId || !$adminId || !$adminName) {
    $msg = "Missing required fields";
    file_put_contents($debugFile, $msg . "\n", FILE_APPEND);
    echo json_encode(['status' => 'error', 'message' => $msg]);
    exit;
}

// 1️⃣ Update order
$updatedRows = updateOrderStatus($orderId, 1, $adminId, $adminName);
file_put_contents($debugFile, "Order updated rows: $updatedRows\n", FILE_APPEND);

// 2️⃣ Log admin action
$logRows = logOrderAction($adminId, 'admin', $adminName, 'approve_order', $orderId, 0, 1);
file_put_contents($debugFile, "Admin log rows: $logRows\n", FILE_APPEND);

// 3️⃣ Save user token
if ($fcmToken) {
    $stmt = $con->prepare("INSERT INTO user_tokens (user_id,fcm_token) VALUES (?,?) ON DUPLICATE KEY UPDATE fcm_token=VALUES(fcm_token), created_at=CURRENT_TIMESTAMP");
    $saved = $stmt->execute([$userId, $fcmToken]);
    file_put_contents($debugFile, "User token saved: " . ($saved ? 'yes' : 'no') . "\n", FILE_APPEND);
}

// 4️⃣ Send FCM notification & insert into DB
try {
    require_once $_SERVER['DOCUMENT_ROOT'] . '/ecommerce/vendor/autoload.php';
    $serviceAccountPath = $_SERVER['DOCUMENT_ROOT'] . '/ecommerce/serviceAccount.json';

    if (!file_exists($serviceAccountPath)) {
        file_put_contents($debugFile, "❌ Firebase service account not found\n", FILE_APPEND);
    } else {
        $factory = (new Kreait\Firebase\Factory)->withServiceAccount($serviceAccountPath);
        $messaging = $factory->createMessaging();

        // Fetch user tokens
        $tokens = [];
        $stmt = $con->prepare("SELECT fcm_token FROM user_tokens WHERE user_id=? AND fcm_token IS NOT NULL AND fcm_token !=''");
        $stmt->execute([$userId]);
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) $tokens[] = $row['fcm_token'];

        if (!empty($tokens)) {
            $title   = "Order Approved ✅";
            $body    = "Your order #$orderId has been approved";
            $titleAr = "تمت الموافقة على الطلب ✅";
            $bodyAr  = "تمت الموافقة على طلبك رقم #$orderId";

            // Insert notification to DB
            $notifStmt = $con->prepare("INSERT INTO user_notifications 
                (notification_title, notification_body, notification_title_ar, notification_body_ar, notification_userid, notification_datetime, is_read) 
                VALUES (?, ?, ?, ?, ?, UTC_TIMESTAMP(), 0)");
            $notifStmt->execute([$title, $body, $titleAr, $bodyAr, $userId]);
            file_put_contents($debugFile, "Notification inserted into DB\n", FILE_APPEND);

            // Send FCM notification
            foreach ($tokens as $token) {
                $message = Kreait\Firebase\Messaging\CloudMessage::withTarget('token', $token)
                    ->withNotification([
                        'title' => $title,
                        'body'  => $body,
                    ])
                    ->withData([
                        'title'     => $title,
                        'body'      => $body,
                        'title_ar'  => $titleAr,
                        'body_ar'   => $bodyAr,
                        'order_id'  => (string)$orderId,
                        'status'    => 'approved',
                        'type'      => 'user_order'
                    ]);

                try {
                    $messaging->send($message);
                    file_put_contents($debugFile, "Notification sent to token: $token\n", FILE_APPEND);
                } catch (Exception $e) {
                    file_put_contents($debugFile, "❌ Failed token $token: " . $e->getMessage() . "\n", FILE_APPEND);
                }
            }
        } else {
            file_put_contents($debugFile, "No user tokens found for user $userId\n", FILE_APPEND);
        }
    }
} catch (Exception $e) {
    file_put_contents($debugFile, "❌ Firebase error: " . $e->getMessage() . "\n", FILE_APPEND);
}

// 5️⃣ Return JSON response
echo json_encode([
    'status'  => $updatedRows > 0 ? 'success' : 'failure',
    'message' => $updatedRows > 0 ? 'Order approved successfully' : 'Order not updated',
    'updated' => $updatedRows,
    'logged'  => $logRows
]);

file_put_contents($debugFile, "----- Approve Script Finished -----\n", FILE_APPEND);
?>
