<?php
// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include the database connection
include "../../connect.php";

// Debug: Log the entire request for troubleshooting
error_log("Received request: " . print_r($_REQUEST, true));

$table = "item_prices";
$id = filterRequest("item_id");
$quantities = isset($_POST['quantities']) ? json_decode($_POST['quantities'], true) : [];
$prices = isset($_POST['prices']) ? json_decode($_POST['prices'], true) : [];

// Debug: Log decoded data
error_log("Decoded item_id: " . $id);
error_log("Decoded quantities: " . json_encode($quantities));
error_log("Decoded prices: " . json_encode($prices));

// Check if quantities and prices are valid and have the same count
if (is_array($quantities) && is_array($prices) && count($quantities) == count($prices)) {
    try {
        // Begin transaction
        $con->beginTransaction();

        for ($i = 0; $i < count($quantities); $i++) {
            $quantity = $quantities[$i];
            $price = $prices[$i];

            // Debug: Log the data before processing
            error_log("Processing item_id: $id with quantity: $quantity, price: $price");

            // Check if the record exists for this item_id and quantity
            $checkSql = "SELECT COUNT(*) FROM $table WHERE item_id = :item_id AND quantity = :quantity";
            $checkStmt = $con->prepare($checkSql);
            $checkStmt->bindParam(':item_id', $id, PDO::PARAM_INT);
            $checkStmt->bindParam(':quantity', $quantity, PDO::PARAM_INT);
            $checkStmt->execute();
            $exists = $checkStmt->fetchColumn();

            if ($price === null || $price === '') {
                // If price is null or empty, delete the record if it exists
                if ($exists) {
                    $deleteSql = "DELETE FROM $table WHERE item_id = :item_id AND quantity = :quantity";
                    $deleteStmt = $con->prepare($deleteSql);
                    $deleteStmt->bindParam(':item_id', $id, PDO::PARAM_INT);
                    $deleteStmt->bindParam(':quantity', $quantity, PDO::PARAM_INT);

                    if (!$deleteStmt->execute()) {
                        throw new Exception("Deletion failed for item_id: $id, quantity: $quantity");
                    }

                    // Debug: Log a success message for deletion
                    error_log("Deletion succeeded for quantity $quantity");
                }
            } else {
                // If price is provided, update or insert based on existence
                if ($exists) {
                    // Update the existing record
                    $updateSql = "UPDATE $table SET price = :price WHERE item_id = :item_id AND quantity = :quantity";
                    $updateStmt = $con->prepare($updateSql);
                    $updateStmt->bindParam(':item_id', $id, PDO::PARAM_INT);
                    $updateStmt->bindParam(':quantity', $quantity, PDO::PARAM_INT);
                    $updateStmt->bindParam(':price', $price, PDO::PARAM_STR);

                    if (!$updateStmt->execute()) {
                        throw new Exception("Update failed for item_id: $id, quantity: $quantity");
                    }

                    // Debug: Log a success message for update
                    error_log("Update succeeded for quantity $quantity");
                } else {
                    // Insert the new record
                    $insertSql = "INSERT INTO $table (item_id, quantity, price) VALUES (:item_id, :quantity, :price)";
                    $insertStmt = $con->prepare($insertSql);
                    $insertStmt->bindParam(':item_id', $id, PDO::PARAM_INT);
                    $insertStmt->bindParam(':quantity', $quantity, PDO::PARAM_INT);
                    $insertStmt->bindParam(':price', $price, PDO::PARAM_STR);

                    if (!$insertStmt->execute()) {
                        throw new Exception("Insert failed for item_id: $id, quantity: $quantity");
                    }

                    // Debug: Log a success message for insert
                    error_log("Insert succeeded for quantity $quantity");
                }
            }
        }

        // Commit transaction
        $con->commit();
        echo json_encode(array("status" => "success"));
    } catch (Exception $e) {
        // Rollback transaction on error
        $con->rollBack();
        error_log("Transaction failed: " . $e->getMessage());
        echo json_encode(array("status" => "failure", "message" => "Database error."));
    }
} else {
    // Debug: Log an error if quantities or prices are invalid
    error_log("Invalid quantities or prices");
    echo json_encode(array("status" => "failure", "message" => "Invalid quantities or prices"));
}
header('Content-Type: application/json'); // Set content type to JSON

?>
