<?php
header('Content-Type: application/json; charset=utf-8');
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

require_once __DIR__ . '/../../connect.php'; // adjust path to your DB connection
// $con should be a PDO instance

// Read input
$raw = file_get_contents('php://input');
$input = json_decode($raw, true) ?: $_POST;

// Required fields
$actor_id   = isset($input['actor_id']) ? (int)$input['actor_id'] : null;
$actor_type = $input['actor_type'] ?? null;
$actor_name = $input['actor_name'] ?? null;
$action_type = $input['action_type'] ?? null;

if (!$actor_id || !$actor_type || !$action_type) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'actor_id, actor_type, and action_type are required']);
    exit;
}

// Optional fields
$reference_table  = $input['reference_table'] ?? null;
$reference_id     = isset($input['reference_id']) ? (int)$input['reference_id'] : null;
$order_total      = isset($input['order_total']) ? (float)$input['order_total'] : null;
$currency         = $input['currency'] ?? 'USD';
$payment_method   = $input['payment_method'] ?? null;
$delivery_method  = $input['delivery_method'] ?? null;
$old_status       = $input['old_status'] ?? null;
$new_status       = $input['new_status'] ?? null;
$details_raw      = $input['details'] ?? null;
$device_info      = $input['device_info'] ?? null;

// Normalize details to JSON
$details_json = null;
if ($details_raw !== null) {
    if (is_array($details_raw) || is_object($details_raw)) {
        $details_json = json_encode($details_raw, JSON_UNESCAPED_UNICODE);
    } else {
        $decoded = json_decode($details_raw, true);
        $details_json = (json_last_error() === JSON_ERROR_NONE) ? json_encode($decoded, JSON_UNESCAPED_UNICODE) : json_encode(['text' => (string)$details_raw], JSON_UNESCAPED_UNICODE);
    }
}

// Server-side IP & user-agent
$user_agent = $_SERVER['HTTP_USER_AGENT'] ?? null;
$ip_addr = $_SERVER['HTTP_CF_CONNECTING_IP'] 
         ?? $_SERVER['HTTP_X_FORWARDED_FOR'] 
         ?? $_SERVER['REMOTE_ADDR'];

// If X-Forwarded-For contains multiple IPs, take first
if (strpos($ip_addr, ',') !== false) {
    $parts = explode(',', $ip_addr);
    $ip_addr = trim($parts[0]);
}

// Insert log
try {
    $sql = "INSERT INTO user_logs 
      (actor_id, actor_type, actor_name, action_type, reference_table, reference_id, order_total, currency, payment_method, delivery_method, old_status, new_status, details, device_info, user_agent, ip_address)
      VALUES (:actor_id, :actor_type, :actor_name, :action_type, :reference_table, :reference_id, :order_total, :currency, :payment_method, :delivery_method, :old_status, :new_status, :details, :device_info, :user_agent, :ip_address)";

    $stmt = $con->prepare($sql);
    $stmt->execute([
        ':actor_id' => $actor_id,
        ':actor_type' => $actor_type,
        ':actor_name' => $actor_name,
        ':action_type' => $action_type,
        ':reference_table' => $reference_table,
        ':reference_id' => $reference_id,
        ':order_total' => $order_total,
        ':currency' => $currency,
        ':payment_method' => $payment_method,
        ':delivery_method' => $delivery_method,
        ':old_status' => $old_status,
        ':new_status' => $new_status,
        ':details' => $details_json,
        ':device_info' => $device_info,
        ':user_agent' => $user_agent,
        ':ip_address' => $ip_addr
    ]);

    echo json_encode(['status' => 'success', 'log_id' => $con->lastInsertId()]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'DB error', 'detail' => $e->getMessage()]);
}
