<?php
header('Content-Type: application/json');
include "connect.php"; // Your existing connection file

try {
    // Get query parameters for filtering
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $limit = isset($_GET['limit']) ? max(1, intval($_GET['limit'])) : 50;
    $offset = ($page - 1) * $limit;
    
    $actor_type = isset($_GET['actor_type']) ? $_GET['actor_type'] : null;
    $action_type = isset($_GET['action_type']) ? $_GET['action_type'] : null;
    $reference_table = isset($_GET['reference_table']) ? $_GET['reference_table'] : null;
    $reference_id = isset($_GET['reference_id']) ? intval($_GET['reference_id']) : null;
    
    // Build base query
    $query = "SELECT * FROM user_logs WHERE 1=1";
    $params = [];
    $types = "";
    
    if ($actor_type) {
        $query .= " AND actor_type = ?";
        $params[] = $actor_type;
        $types .= "s";
    }
    
    if ($action_type) {
        $query .= " AND action_type = ?";
        $params[] = $action_type;
        $types .= "s";
    }
    
    if ($reference_table) {
        $query .= " AND reference_table = ?";
        $params[] = $reference_table;
        $types .= "s";
    }
    
    if ($reference_id) {
        $query .= " AND reference_id = ?";
        $params[] = $reference_id;
        $types .= "i";
    }
    
    // Get total count
    $count_query = "SELECT COUNT(*) as total FROM user_logs WHERE 1=1";
    $count_params = [];
    $count_types = "";
    
    if ($actor_type) {
        $count_query .= " AND actor_type = ?";
        $count_params[] = $actor_type;
        $count_types .= "s";
    }
    if ($action_type) {
        $count_query .= " AND action_type = ?";
        $count_params[] = $action_type;
        $count_types .= "s";
    }
    if ($reference_table) {
        $count_query .= " AND reference_table = ?";
        $count_params[] = $reference_table;
        $count_types .= "s";
    }
    if ($reference_id) {
        $count_query .= " AND reference_id = ?";
        $count_params[] = $reference_id;
        $count_types .= "i";
    }
    
    $count_stmt = $con->prepare($count_query);
    if (!empty($count_params)) {
        $count_stmt->execute($count_params);
    } else {
        $count_stmt->execute();
    }
    $total_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
    $total_logs = $total_result['total'];
    
    // FIXED: Add LIMIT and OFFSET directly to the query (not as parameters)
    $query .= " ORDER BY created_at DESC LIMIT $limit OFFSET $offset";
    
    $stmt = $con->prepare($query);
    
    // Bind parameters only if they exist
    if (!empty($params)) {
        $stmt->execute($params);
    } else {
        $stmt->execute();
    }
    
    $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Process each log to add human-readable information
    foreach ($logs as &$log) {
        $targetName = "Unknown Target";
        $deliveryMethodLabel = null;
        $paymentMethodLabel = null;

        switch ($log['reference_table']) {
            case 'users':
                $stmt2 = $con->prepare("SELECT users_name FROM users WHERE users_id = ?");
                $stmt2->execute([$log['reference_id']]);
                $targetName = $stmt2->fetchColumn() ?: "Deleted User #".$log['reference_id'];
                break;

            case 'admin':
                $stmt2 = $con->prepare("SELECT admin_name FROM admin WHERE admin_id = ?");
                $stmt2->execute([$log['reference_id']]);
                $targetName = $stmt2->fetchColumn() ?: "Deleted Admin #".$log['reference_id'];
                break;

            case 'delivery':
                $stmt2 = $con->prepare("SELECT delivery_name FROM delivery WHERE delivery_id = ?");
                $stmt2->execute([$log['reference_id']]);
                $targetName = $stmt2->fetchColumn() ?: "Deleted Delivery #".$log['reference_id'];
                break;

            case 'orders':
                $stmt2 = $con->prepare("SELECT orders_type, orders_paymentmethod FROM orders WHERE orders_id = ?");
                $stmt2->execute([$log['reference_id']]);
                $order = $stmt2->fetch(PDO::FETCH_ASSOC);

                if ($order) {
                    // delivery method
                    $deliveryMethodLabel = ($order['orders_type'] == 0) ? "Delivery" : "Receive";

                    // payment method
                    if ($order['orders_paymentmethod'] == 0) {
                        $paymentMethodLabel = "Cash";
                    } else {
                        $paymentMethodLabel = "Card";
                    }
                }

                $targetName = "Order #".$log['reference_id'];
                break;
                
            default:
                $targetName = $log['reference_table'] . " #" . $log['reference_id'];
                break;
        }

        // Add processed fields
        $log['target_name'] = $targetName;
        if ($deliveryMethodLabel !== null) $log['delivery_method_label'] = $deliveryMethodLabel;
        if ($paymentMethodLabel !== null) $log['payment_method_label'] = $paymentMethodLabel;
        
        // Decode JSON values if they exist
        if ($log['old_values'] && $log['old_values'] !== 'null') {
            $log['old_values'] = json_decode($log['old_values'], true);
        } else {
            $log['old_values'] = null;
        }
        
        if ($log['new_values'] && $log['new_values'] !== 'null') {
            $log['new_values'] = json_decode($log['new_values'], true);
        } else {
            $log['new_values'] = null;
        }
    }

    // Return JSON response
    echo json_encode([
        'status' => 'success',
        'page' => $page,
        'limit' => $limit,
        'total' => $total_logs,
        'total_pages' => ceil($total_logs / $limit),
        'data' => $logs
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => 'Database error: ' . $e->getMessage()
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => 'Server error: ' . $e->getMessage()
    ]);
}
?>