<?php
// -------------------- CONFIG --------------------
date_default_timezone_set('Europe/Berlin'); // Germany time zone

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8'); // Ensure UTF-8 JSON output

// Set up logging in the same directory as checkout.php
$log_directory = __DIR__;
$error_log_file = $log_directory . '/checkout_errors.log';
$debug_log_file = $log_directory . '/checkout_debug.log';

// Initialize response array
$response = ["status" => "failure", "message" => "Unknown error occurred"];

// Log start of process
error_log("=== CHECKOUT PROCESS STARTED ===" . PHP_EOL, 3, $error_log_file);
error_log("Timestamp: " . date('Y-m-d H:i:s') . PHP_EOL, 3, $error_log_file);
error_log("POST Data: " . json_encode($_POST) . PHP_EOL, 3, $debug_log_file);
error_log("GET Data: " . json_encode($_GET) . PHP_EOL, 3, $debug_log_file);

include "../connect.php"; // DB connection

// Check DB connection
if (!$con) {
    $msg = "❌ DATABASE CONNECTION FAILED!";
    error_log($msg . PHP_EOL, 3, $error_log_file);
    $response = ["status" => "failure", "message" => $msg];
    goto output_response;
}

// -------------------- HELPER --------------------
if (!function_exists('filterRequest')) {
    function filterRequest($key) {
        return $_POST[$key] ?? $_GET[$key] ?? null;
    }
}

// -------------------- GET DATA --------------------
$usersid = filterRequest("usersid");
$orderstype = filterRequest("orderstype");
$paymentmethod = filterRequest("paymentmethod");
$ordersprice = filterRequest("ordersprice");
$pricedelivery = filterRequest("pricedelivery");
$addressid = filterRequest("addressid");
$couponid = filterRequest("couponid");
$coupondiscount = filterRequest("coupondiscount");

// Log received data
error_log("Received data - usersid: $usersid, orderstype: $orderstype, paymentmethod: $paymentmethod, ordersprice: $ordersprice, pricedelivery: $pricedelivery, addressid: $addressid, couponid: $couponid, coupondiscount: $coupondiscount" . PHP_EOL, 3, $debug_log_file);

// Validate required fields - FIXED VERSION
// Check if values are set, not if they're truthy (0 is a valid value)
if (!isset($usersid) || !isset($orderstype) || !isset($paymentmethod) || !isset($ordersprice) ||
    $usersid === '' || $orderstype === '' || $paymentmethod === '' || $ordersprice === '') {
    $msg = "❌ Missing required fields: usersid=$usersid, orderstype=$orderstype, paymentmethod=$paymentmethod, ordersprice=$ordersprice";
    error_log($msg . PHP_EOL, 3, $error_log_file);
    $response = ["status" => "failure", "message" => $msg];
    goto output_response;
}

// Additional validation for numeric values
if (!is_numeric($ordersprice) || floatval($ordersprice) <= 0) {
    $msg = "❌ Invalid order price: $ordersprice";
    error_log($msg . PHP_EOL, 3, $error_log_file);
    $response = ["status" => "failure", "message" => $msg];
    goto output_response;
}

if ($orderstype == "1") $pricedelivery = 0; // Pickup

$totalprice = ($ordersprice + $pricedelivery) * 1.15; // 15% tax

// -------------------- HANDLE COUPON --------------------
$now = date("Y-m-d H:i:s");
if ($couponid && $couponid != "0") {
    error_log("Processing coupon: $couponid" . PHP_EOL, 3, $debug_log_file);
    $stmt = $con->prepare("SELECT coupon_count, coupon_expiredate FROM coupon WHERE coupon_id = ? LIMIT 1");
    $stmt->execute([$couponid]);
    $coupon = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($coupon) {
        error_log("Coupon found: " . json_encode($coupon) . PHP_EOL, 3, $debug_log_file);
        if ($coupon['coupon_expiredate'] > $now && $coupon['coupon_count'] > 0) {
            $discountAmount = $ordersprice * $coupondiscount / 100;
            $totalprice -= $discountAmount;
            error_log("Applying coupon discount: $discountAmount, New total: $totalprice" . PHP_EOL, 3, $debug_log_file);
            
            $updateStmt = $con->prepare("UPDATE coupon SET coupon_count = coupon_count - 1 WHERE coupon_id = ?");
            $updateStmt->execute([$couponid]);
            error_log("Coupon count updated for ID: $couponid" . PHP_EOL, 3, $debug_log_file);
        } else {
            error_log("Coupon expired or out of stock: expiredate={$coupon['coupon_expiredate']}, count={$coupon['coupon_count']}" . PHP_EOL, 3, $debug_log_file);
        }
    } else {
        error_log("Coupon not found: $couponid" . PHP_EOL, 3, $debug_log_file);
    }
}

// -------------------- INSERT ORDER --------------------
$orderData = [
    "orders_usersid" => $usersid,
    "orders_type" => $orderstype,
    "orders_paymentmethod" => $paymentmethod,
    "orders_price" => $ordersprice,
    "orders_pricedelivery" => $pricedelivery,
    "orders_address" => $addressid,
    "orders_coupon" => $couponid,
    "orders_totalprice" => $totalprice,
];

error_log("Inserting order data: " . json_encode($orderData) . PHP_EOL, 3, $debug_log_file);

$count = insertData("orders", $orderData, false);

if ($count > 0) {
    error_log("Order inserted successfully, count: $count" . PHP_EOL, 3, $debug_log_file);
    
    $stmt = $con->prepare("SELECT MAX(orders_id) AS maxid FROM orders");
    $stmt->execute();
    $maxid = $stmt->fetchColumn();
    error_log("Max order ID retrieved: $maxid" . PHP_EOL, 3, $debug_log_file);

    // Update cart
    $cartUpdate = updateData("cart", ["cart_orders" => $maxid], "cart_usersid = $usersid AND cart_orders = 0", false);
    error_log("Cart update result: " . ($cartUpdate ? "success" : "failed") . PHP_EOL, 3, $debug_log_file);

    // Send notifications
    $notifResult = sendNotificationToAdmins($maxid, $totalprice, $usersid, $addressid, $paymentmethod, $orderstype);
    error_log("Notification result: " . ($notifResult ? "success" : "failed") . PHP_EOL, 3, $debug_log_file);

    $response = [
        "status" => "success",
        "message" => "✅ تم الطلب بنجاح! رقم الطلب: $maxid",
        "order_id" => $maxid,
        "notification_sent" => $notifResult
    ];
    
    error_log("Order completed successfully: $maxid" . PHP_EOL, 3, $debug_log_file);
} else {
    $msg = "❌ فشل في إدخال الطلب. تحقق من الحقول المطلوبة أو هيكل قاعدة البيانات.";
    error_log($msg . PHP_EOL, 3, $error_log_file);
    $response = [
        "status" => "failure",
        "message" => $msg
    ];
}

// -------------------- FUNCTIONS --------------------
function sendNotificationToAdmins($orderId, $totalPrice, $userId, $addressId, $paymentMethod, $orderType) {
    global $con, $error_log_file, $debug_log_file;

    try {
        error_log("Sending notification for order: $orderId" . PHP_EOL, 3, $debug_log_file);
        
        require_once $_SERVER['DOCUMENT_ROOT'] . '/ecommerce/vendor/autoload.php';
        $serviceAccountPath = $_SERVER['DOCUMENT_ROOT'] . '/ecommerce/serviceAccount.json';
        
        if (!file_exists($serviceAccountPath)) {
            error_log("❌ Firebase service account file not found: $serviceAccountPath" . PHP_EOL, 3, $error_log_file);
            return false;
        }
        
        $factory = (new Kreait\Firebase\Factory)->withServiceAccount($serviceAccountPath);
        $messaging = $factory->createMessaging();

        // Admin tokens
        $adminTokens = [];
        $stmt = $con->prepare("SELECT fcm_token FROM admin_tokens WHERE fcm_token IS NOT NULL AND fcm_token != ''");
        $stmt->execute();
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) $adminTokens[] = $row['fcm_token'];

        error_log("Found " . count($adminTokens) . " admin tokens" . PHP_EOL, 3, $debug_log_file);

        if (empty($adminTokens)) {
            error_log("No admin tokens found" . PHP_EOL, 3, $debug_log_file);
            return false;
        }

        // Enhanced notification details
        $itemCount = getOrderItemCount($orderId);
        $addressText = getAddressText($addressId);
        $paymentText = getPaymentMethodText($paymentMethod);
        $orderTypeText = getOrderTypeText($orderType);
        $customerName = getCustomerName($userId);
        $customerPhone = getCustomerPhone($userId);

        $title = "🛍️ Alwasat Company - طلب جديد #$orderId";
        $body = "👤 $customerName\n📞 $customerPhone\n💵 الإجمالي: " . number_format($totalPrice, 2) . " ر.س\n📍 $addressText\n🛒 المنتجات: $itemCount منتج\n📦 $orderTypeText\n🎯 $paymentText\n⏰ " . date('M j, g:i A');

        $message = Kreait\Firebase\Messaging\CloudMessage::new()
            ->withNotification([
                'title' => $title,
                'body' => $body,
                'sound' => 'default',
                'android_channel_id' => 'order_notifications'
            ])
            ->withData([
                'order_id' => (string)$orderId,
                'total_price' => (string)$totalPrice,
                'type' => 'new_order',
                'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                 'timestamp' => time(), // Unique identifier for each notification
            ]);

        $successCount = 0;
        foreach ($adminTokens as $token) {
            try {
                $messaging->send($message->withChangedTarget('token', $token));
                $successCount++;
                error_log("Notification sent to token: $token" . PHP_EOL, 3, $debug_log_file);
            } catch (Exception $e) {
                error_log("❌ Failed to send notification to token: $token - Error: " . $e->getMessage() . PHP_EOL, 3, $error_log_file);
                if (strpos($e->getMessage(), 'UNREGISTERED')!==false || strpos($e->getMessage(), 'NOT_FOUND')!==false) {
                    $delStmt = $con->prepare("DELETE FROM admin_tokens WHERE fcm_token = ?");
                    $delStmt->execute([$token]);
                    error_log("Removed invalid token: $token" . PHP_EOL, 3, $debug_log_file);
                }
            }
        }

        $notifData = [
            "title" => $title,
            "body" => $body,
            "title_ar" => $title,
            "body_ar" => $body,
            "user_id" => $userId,
            "is_read" => 0,
            "order_id" => $orderId
        ];
        $notifInsert = insertData("admin_notifications", $notifData, false);
        error_log("Notification data inserted: " . ($notifInsert ? "success" : "failed") . PHP_EOL, 3, $debug_log_file);

        error_log("Notifications sent successfully: $successCount/" . count($adminTokens) . PHP_EOL, 3, $debug_log_file);
        return $successCount > 0;
    } catch (Exception $e) {
        error_log("❌ Firebase error: " . $e->getMessage() . PHP_EOL, 3, $error_log_file);
        return false;
    }
}

// Helper functions for enhanced notifications
function getOrderItemCount($orderId) {
    global $con;
    $stmt = $con->prepare("SELECT COUNT(*) as count FROM cart WHERE cart_orders = ?");
    $stmt->execute([$orderId]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result['count'] ?? 0;
}

function getAddressText($addressId) {
    global $con;
    $stmt = $con->prepare("SELECT address_name, address_street FROM address WHERE address_id = ?");
    $stmt->execute([$addressId]);
    $address = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($address) return $address['address_name'] . ": " . substr($address['address_street'], 0, 20) . "...";
    return "العنوان #" . $addressId;
}

function getCustomerName($userId) {
    global $con;
    $stmt = $con->prepare("SELECT users_name FROM users WHERE users_id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($user) return trim($user['users_name']) ?: "العميل #$userId";
    return "العميل #$userId";
}

function getCustomerPhone($userId) {
    global $con;
    $stmt = $con->prepare("SELECT users_phone FROM users WHERE users_id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    return $user['users_phone'] ?? '';
}

function getPaymentMethodText($methodCode) {
    $methods = ['0'=>'💵 الدفع نقداً','1'=>'💳 الدفع بالبطاقة','2'=>'💳 الدفع بالبطاقة'];
    return $methods[$methodCode] ?? '💳 طريقة الدفع ' . $methodCode;
}

function getOrderTypeText($typeCode) {
    $types = ['0'=>'🚚 توصيل','1'=>'🏪 استلام'];
    return $types[$typeCode] ?? '📦 نوع الطلب ' . $typeCode;
}

// Output the response (ONLY ONCE)
output_response:
error_log("Final response: " . json_encode($response) . PHP_EOL, 3, $debug_log_file);
error_log("=== CHECKOUT PROCESS COMPLETED ===" . PHP_EOL . PHP_EOL, 3, $debug_log_file);
echo json_encode($response);
exit;