<?php
// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8');

// Log file for debugging
$log_file = __DIR__ . '/delete_debug.log';
file_put_contents($log_file, "=== DELETE PROCESS STARTED ===\n", FILE_APPEND);
file_put_contents($log_file, "Time: " . date('Y-m-d H:i:s') . "\n", FILE_APPEND);

// Get parameters directly
$ordersid = $_REQUEST['id'] ?? '';
$userid = $_REQUEST['userid'] ?? '';

file_put_contents($log_file, "Received - Order: $ordersid, User: $userid\n", FILE_APPEND);

// Validate parameters
if (empty($ordersid) || empty($userid)) {
    file_put_contents($log_file, "Missing parameters\n", FILE_APPEND);
    echo json_encode(["status" => "failure", "message" => "❌ Missing parameters"]);
    exit;
}

// Include database connection
include "../connect.php";

// Check database connection
if (!$con) {
    file_put_contents($log_file, "Database connection failed\n", FILE_APPEND);
    echo json_encode(["status" => "failure", "message" => "❌ Database error"]);
    exit;
}

file_put_contents($log_file, "Database connected successfully\n", FILE_APPEND);

try {
    // Check if order exists and belongs to user
    $stmt = $con->prepare("SELECT orders_id, orders_usersid, orders_status, orders_totalprice FROM orders WHERE orders_id = ?");
    $stmt->execute([$ordersid]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);

    file_put_contents($log_file, "Order query result: " . json_encode($order) . "\n", FILE_APPEND);

    if (!$order) {
        file_put_contents($log_file, "Order not found\n", FILE_APPEND);
        echo json_encode(["status" => "failure", "message" => "❌ الطلبية غير موجودة"]);
        exit;
    }

    // Check if order belongs to user
    if ($order['orders_usersid'] != $userid) {
        file_put_contents($log_file, "Order belongs to different user\n", FILE_APPEND);
        echo json_encode(["status" => "failure", "message" => "❌ ليس لديك صلاحية حذف هذه الطلبية"]);
        exit;
    }

    // Check if order status is 0 (pending)
    if ($order['orders_status'] != 0) {
        file_put_contents($log_file, "Order status is not 0: {$order['orders_status']}\n", FILE_APPEND);
        echo json_encode(["status" => "failure", "message" => "❌ لا يمكن حذف الطلبية لأنها ليست قيد الانتظار"]);
        exit;
    }

    // Get user name for the deleted_by_name field
    $user_stmt = $con->prepare("SELECT users_name, users_phone FROM users WHERE users_id = ?");
    $user_stmt->execute([$userid]);
    $user = $user_stmt->fetch(PDO::FETCH_ASSOC);
    $username = $user['users_name'] ?? "User $userid";
    $userphone = $user['users_phone'] ?? "";
    
    // Instead of deleting, update the order status to "cancelled" and set deleted_by fields
    $update_stmt = $con->prepare("UPDATE orders 
                                 SET orders_status = 4, 
                                     deleted_by_id = ?, 
                                     deleted_by_name = ?,
                                     orders_datetime = NOW()
                                 WHERE orders_id = ? AND orders_usersid = ?");
    
    $update_stmt->execute([$userid, $username, $ordersid, $userid]);
    $updated_count = $update_stmt->rowCount();

    file_put_contents($log_file, "Update affected rows: $updated_count\n", FILE_APPEND);

    if ($updated_count > 0) {
        file_put_contents($log_file, "Order cancelled successfully\n", FILE_APPEND);
        
        // ------------------- SEND NOTIFICATION TO ADMINS -------------------
        $notification_sent = sendNotificationToAdmins($ordersid, $order['orders_totalprice'], $userid, $username, $userphone);
        file_put_contents($log_file, "Notification sent: " . ($notification_sent ? "YES" : "NO") . "\n", FILE_APPEND);
        // -------------------------------------------------------------------
        
        echo json_encode([
            "status" => "success", 
            "message" => "✅ تم إلغاء الطلبية بنجاح",
            "order_id" => $ordersid,
            "notification_sent" => $notification_sent
        ]);
    } else {
        file_put_contents($log_file, "Update failed - no rows affected\n", FILE_APPEND);
        echo json_encode(["status" => "failure", "message" => "❌ فشل في إلغاء الطلبية"]);
    }

} catch (Exception $e) {
    file_put_contents($log_file, "Exception: " . $e->getMessage() . "\n", FILE_APPEND);
    echo json_encode(["status" => "failure", "message" => "❌ Server error: " . $e->getMessage()]);
}

// ------------------- NOTIFICATION FUNCTION -------------------
function sendNotificationToAdmins($orderId, $totalPrice, $userId, $userName, $userPhone) {
    global $con, $log_file;
    
    file_put_contents($log_file, "Preparing to send notification for order: $orderId\n", FILE_APPEND);
    
    try {
        // Check if Firebase library exists
        $firebasePath = $_SERVER['DOCUMENT_ROOT'] . '/ecommerce/vendor/autoload.php';
        if (!file_exists($firebasePath)) {
            file_put_contents($log_file, "Firebase library not found: $firebasePath\n", FILE_APPEND);
            return false;
        }
        
        require_once $firebasePath;
        
        $serviceAccountPath = $_SERVER['DOCUMENT_ROOT'] . '/ecommerce/serviceAccount.json';
        if (!file_exists($serviceAccountPath)) {
            file_put_contents($log_file, "Service account file not found: $serviceAccountPath\n", FILE_APPEND);
            return false;
        }
        
        $factory = (new Kreait\Firebase\Factory)->withServiceAccount($serviceAccountPath);
        $messaging = $factory->createMessaging();

        // Get admin tokens
        $adminTokens = [];
        $stmt = $con->prepare("SELECT fcm_token FROM admin_tokens WHERE fcm_token IS NOT NULL AND fcm_token != ''");
        $stmt->execute();
        
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $adminTokens[] = $row['fcm_token'];
        }

        file_put_contents($log_file, "Found " . count($adminTokens) . " admin tokens\n", FILE_APPEND);

        if (empty($adminTokens)) {
            file_put_contents($log_file, "No admin tokens found\n", FILE_APPEND);
            return false;
        }

        // Prepare notification message
        $title = "⚠️ تم إلغاء الطلبية #$orderId";
        $body = "تم إلغاء الطلبية رقم #$orderId من قبل العميل $userName ($userPhone) - المبلغ: " . number_format($totalPrice, 2) . " ر.س";

        $message = Kreait\Firebase\Messaging\CloudMessage::new()
            ->withNotification([
                'title' => $title,
                'body' => $body,
                'sound' => 'default'
            ])
            ->withData([
                'order_id' => (string)$orderId,
                'type' => 'order_cancelled',
                'click_action' => 'FLUTTER_NOTIFICATION_CLICK'
            ]);

        $successCount = 0;
        foreach ($adminTokens as $token) {
            try {
                $messaging->send($message->withChangedTarget('token', $token));
                $successCount++;
                file_put_contents($log_file, "Notification sent to token: $token\n", FILE_APPEND);
            } catch (Exception $e) {
                file_put_contents($log_file, "Failed to send to token $token: " . $e->getMessage() . "\n", FILE_APPEND);
                
                // Remove invalid tokens
                if (strpos($e->getMessage(), 'UNREGISTERED') !== false || strpos($e->getMessage(), 'NOT_FOUND') !== false) {
                    $deleteStmt = $con->prepare("DELETE FROM admin_tokens WHERE fcm_token = ?");
                    $deleteStmt->execute([$token]);
                    file_put_contents($log_file, "Removed invalid token: $token\n", FILE_APPEND);
                }
            }
        }

        // Insert notification into database
        $notifData = [
            "title" => $title,
            "body" => $body,
            "title_ar" => $title,
            "body_ar" => $body,
            "user_id" => $userId,
            "is_read" => 0,
            "order_id" => $orderId
        ];
        
        // Use direct SQL insert to avoid function issues
        $insertStmt = $con->prepare("INSERT INTO admin_notifications (title, body, title_ar, body_ar, user_id, is_read, order_id) 
                                    VALUES (?, ?, ?, ?, ?, ?, ?)");
        $insertStmt->execute([
            $notifData['title'],
            $notifData['body'],
            $notifData['title_ar'],
            $notifData['body_ar'],
            $notifData['user_id'],
            $notifData['is_read'],
            $notifData['order_id']
        ]);
        
        file_put_contents($log_file, "Notification inserted into database\n", FILE_APPEND);
        file_put_contents($log_file, "Notifications sent successfully: $successCount/" . count($adminTokens) . "\n", FILE_APPEND);
        
        return $successCount > 0;
        
    } catch (Exception $e) {
        file_put_contents($log_file, "Notification error: " . $e->getMessage() . "\n", FILE_APPEND);
        return false;
    }
}
?>