<?php
// Correct path from orders/notifications.php to ecommerce/connect.php
include "../connect.php";

// Debug: Check if connection is successful
if (!$con) {
    echo json_encode(array("status" => "error", "message" => "Database connection failed"));
    exit;
}

$action = filterRequest("action");

// Debug: Check if action parameter is received
if (!$action) {
    echo json_encode(array("status" => "error", "message" => "No action specified"));
    exit;
}

switch ($action) {
    case 'get_notifications':
        getNotifications();
        break;
    case 'mark_as_read':
        markAsRead();
        break;
    case 'mark_all_read':
        markAllAsRead();
        break;
    case 'get_unread_count':
        getUnreadCount();
        break;
    default:
        echo json_encode(array("status" => "error", "message" => "Invalid action: " . $action));
}

function getNotifications() {
    global $con;
    
    // Get parameters and convert to integers
    $page = (int)(isset($_POST['page']) ? $_POST['page'] : 1);
    $limit = (int)(isset($_POST['limit']) ? $_POST['limit'] : 10);
    $offset = ($page - 1) * $limit;
    
    // Debug: Check values
    error_log("Page: $page, Limit: $limit, Offset: $offset");
    
    try {
        $stmt = $con->prepare("
            SELECT * FROM admin_notifications 
            ORDER BY created_at DESC 
            LIMIT :limit OFFSET :offset
        ");
        
        // Bind parameters as integers
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        
        $stmt->execute();
        $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode(array(
            "status" => "success",
            "data" => $notifications,
            "page" => $page,
            "limit" => $limit,
            "total" => count($notifications)
        ));
        
    } catch (PDOException $e) {
        echo json_encode(array(
            "status" => "error", 
            "message" => "Database error: " . $e->getMessage()
        ));
    }
}

function markAsRead() {
    global $con;
    
    $notificationId = filterRequest("notification_id");
    
    if (!$notificationId) {
        echo json_encode(array("status" => "error", "message" => "No notification ID provided"));
        return;
    }
    
    try {
        $stmt = $con->prepare("
            UPDATE admin_notifications 
            SET is_read = 1 
            WHERE id = ?
        ");
        $success = $stmt->execute([$notificationId]);
        
        if ($success) {
            echo json_encode(array(
                "status" => "success",
                "message" => "Notification marked as read"
            ));
        } else {
            echo json_encode(array(
                "status" => "error",
                "message" => "Failed to mark notification as read"
            ));
        }
    } catch (PDOException $e) {
        echo json_encode(array(
            "status" => "error", 
            "message" => "Database error: " . $e->getMessage()
        ));
    }
}

function markAllAsRead() {
    global $con;
    
    try {
        $stmt = $con->prepare("
            UPDATE admin_notifications 
            SET is_read = 1 
            WHERE is_read = 0
        ");
        $success = $stmt->execute();
        
        if ($success) {
            echo json_encode(array(
                "status" => "success",
                "message" => "All notifications marked as read"
            ));
        } else {
            echo json_encode(array(
                "status" => "error",
                "message" => "Failed to mark all notifications as read"
            ));
        }
    } catch (PDOException $e) {
        echo json_encode(array(
            "status" => "error", 
            "message" => "Database error: " . $e->getMessage()
        ));
    }
}

function getUnreadCount() {
    global $con;
    
    try {
        $stmt = $con->prepare("
            SELECT COUNT(*) as unread_count 
            FROM admin_notifications 
            WHERE is_read = 0
        ");
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Convert to integer before returning
        $unreadCount = (int)$result['unread_count'];
        
        echo json_encode(array(
            "status" => "success",
            "unread_count" => $unreadCount  // Now it's integer, not string
        ));
    } catch (PDOException $e) {
        echo json_encode(array(
            "status" => "error", 
            "message" => "Database error: " . $e->getMessage()
        ));
    }
}
